(*********************************************
TCustomGridSpriteSurface->TSpriteSurface

Provides a means to access a TSpriteSurface with
a superimposed grid.  Properties determine whether:

o The current Cursor location is rendered
o The current Mouse position is rendered
o A Grid is superimposed over the surface

Descendant classes must override the abstract
methods to provide different types of grids.
See the example TSquareGridSpriteSurface.
*********************************************)

unit GridSpriteSurface;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  TurboSprite, DsgnIntf;

type

  TMultiSelectEvent = procedure( Sender: TObject; Sprites: TList ) of object;

  TCustomGridSpriteSurface = class( TSpriteSurface )
  private
    procedure MyAfterSpriteRender( Sender: TObject );
  protected
    FMouseDown: TMouseEvent;
    FMouseMove: TMouseMoveEvent;
    FMouseUp: TMouseEvent;
    StartX, StartY: integer;
    MouseX, MouseY: integer;
    Dragging: boolean;
    FOnArea: TMultiSelectEvent;
    FOnDeselect: TNotifyEvent;
    FCursorColor: byte;
    FMouseColor: byte;
    FCursX, FCursY: integer;
    FShowMouse: boolean;
    FCellsX: integer;
    FCellsY: integer;
    RatioX: integer;
    RatioY: integer;
    FShowCursor: boolean;
    lstArea: TList;
    OldAfterSpriteRender: TNotifyEvent;
    rectSelected: TRect;
    FShowGrid: boolean;
    FGridColor: byte;
    FSelectBox: byte;
    procedure MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); override;
    procedure MouseMove( Shift: TShiftState; X, Y: Integer ); override;
    procedure MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); override;
    procedure translateCoords( var X: integer; var Y: integer ); virtual; abstract;
    procedure Loaded; override;
    procedure SelectArea; virtual;
    procedure setCellsX( n: integer ); virtual;
    procedure setCellsY( n: integer ); virtual;
    procedure drawCursor; virtual; abstract;
    procedure drawMouse; virtual; abstract;
    procedure drawGrid; virtual; abstract;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
    property CursorX: integer read FCursX write FCursX;
    property CursorY: integer read FCursY write FCursY;
  published
    property CellsX: integer read FCellsX write setCellsX;
    property CellsY: integer read FCellsY write setCellsY;
    property CursorColorIndex: byte read FCursorColor write FCursorColor;
    property MouseColorIndex: byte read FMouseColor write FMouseColor;
    property GridColorIndex: byte read FGridColor write FGridColor;
    property SelectBoxColorIndex: byte read FSelectBox write FSelectBox default 255;
    property ShowCursor: boolean read FShowCursor write FShowCursor default true;
    property ShowMouse: boolean read FShowMouse write FShowMouse default true;
    property ShowGrid: boolean read FShowGrid write FShowGrid;
    property OnMouseDown: TMouseEvent read FMouseDown write FMouseDown;
    property OnMouseMove: TMouseMoveEvent read FMouseMove write FMouseMove;
    property OnMouseUp: TMouseEvent read FMouseUp write FMouseUp;
    property OnAreaSelected: TMultiSelectEvent read FOnArea write FOnArea;
    property OnAreaDeselected: TNotifyEvent read FOnDeselect write FOnDeselect;
  end;

  TSquareGridSpriteSurface = class( TCustomGridSpriteSurface )
  private
  protected
    procedure translateCoords( var X: integer; var Y: integer ); override;
    procedure drawCursor; override;
    procedure drawMouse; override;
    procedure drawGrid; override;
  public
  published
  end;

procedure Register;

implementation

constructor TCustomGridSpriteSurface.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  lstArea := TList.Create;
  FShowMouse := true;
  FShowCursor := true;
  FSelectBox := 255;
end;

destructor TCustomGridSpriteSurface.Destroy;
begin
  lstArea.Free;
  inherited Destroy;
end;

procedure TCustomGridSpriteSurface.Loaded;
begin
  inherited Loaded;
  if not (csDesigning in ComponentState) then
  begin
    OldAfterSpriteRender := FAfterSpriteRender;
    FAfterSpriteRender := MyAfterSpriteRender;
  end;
end;

procedure TCustomGridSpriteSurface.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: integer );
begin
  Dragging := true;
  StartX := getMouseX(X);
  StartY := getMouseY(Y);
  translateCoords( StartX, StartY );
  if Assigned( FMouseDown ) then
    FMouseDown( self, Button, Shift, StartX, StartY );
end;

procedure TCustomGridSpriteSurface.MouseMove( Shift: TShiftState; X, Y: Integer );
begin
  MouseX := getMouseX(X);
  MouseY := getMouseY(Y);
  translateCoords( MouseX, MouseY );
  if Assigned( FAbsMouseMove ) then
    FAbsMouseMove( self, Shift, X, Y );
  if Assigned( FMouseMove ) then
    FMouseMove( self, Shift, MouseX, MouseY );
end;

procedure TCustomGridSpriteSurface.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: integer );
var
  i, n, s: integer;
  en: TSpriteEngine;
  spr: TSprite;
  rect: TRect;
  pt: TPoint;
begin
  Dragging := false;
  FCursX := getMouseX( X );
  FCursY := getMouseY( Y );
  pt.X := FCursX;
  pt.Y := FCursY;
  translateCoords( FCursX, FCursY );
  if Assigned( FMouseUp ) then
    FMouseUp( self, Button, Shift, FCursX, FCursY );
  if (StartX <> MouseX) or (StartY <> MouseY) then
  begin
    lstArea.Clear;
    SelectArea;
    if Assigned( FOnArea ) then
      FOnArea( self, lstArea );
    Exit;
  end;
  if not ( ssShift in Shift ) then
    if Assigned( FOnDeselect ) then
      FOnDeselect( self );
  if Assigned( FSpriteClicked ) then
  begin
    for i := 0 to lstEngines.Count - 1 do
    begin
      en := TSpriteEngine( lstEngines[i] );
      if not en.ClickSprites then
        continue;
      n := en.SpriteCount;
      for s := 0 to n - 1 do
      begin
        spr := en.Sprite[s];
        rect := spr.BoundingRect;
        OffsetRect( rect, OffsetX, OffsetY );
        if PtInRect( rect, pt ) then
        begin
          FSpriteClicked( self, Button, Shift, spr, FCursX, FCursY );
          Exit;
        end;
      end;
    end;
    FSpriteClicked( self, Button, Shift, nil, FCursX, FCursY );
  end;
end;

procedure TCustomGridSpriteSurface.SelectArea;
var
  i, n, ss: integer;
  se: TSpriteEngine;
  sprite: TSprite;
  rectIntersect: TRect;
begin
  for i := 0 to lstEngines.Count - 1 do
  begin
    se := TSpriteEngine( lstEngines[i] );
    ss := se.SpriteCount;
    for n := 0 to ss - 1 do
    begin
      sprite := se.Sprite[n];
      if IntersectRect( rectIntersect, rectSelected, sprite.BoundingRect ) then
        lstArea.Add( sprite );
    end;
  end;
end;

procedure TCustomGridSpriteSurface.setCellsX( n: integer );
begin
  FCellsX := n;
  if n <> 0 then
    RatioX := LogicalWidth div n;
end;

procedure TCustomGridSpriteSurface.setCellsY( n: integer );
begin
  FCellsY := n;
  if n <> 0 then
    RatioY := LogicalHeight div n;
end;

procedure TCustomGridSpriteSurface.MyAfterSpriteRender;
var
  DrawX, DrawY: integer;
  DragX, DragY: integer;
  X1, Y1, X2, Y2: integer;
begin
  if Assigned( OldAfterSpriteRender ) then
    OldAfterSpriteRender( self );
  if FShowGrid then
  begin
    DIBCanvas.PenColorIndex := FMouseColor;
    drawGrid;
  end;
  if FShowMouse then
  begin
    DIBCanvas.PenColorIndex := FMouseColor;
    drawMouse;
  end;
  if FShowCursor then
  begin
    DIBCanvas.PenColorIndex := FCursorColor;
    drawCursor;
  end;
  if Dragging then
  begin
    if (MouseX <> StartX) or (MouseY <> StartY) then
    begin
      with DIBCanvas do
      begin
        PenColorIndex := FSelectBox;
        if MouseX >= StartX then
        begin
          X1 := StartX;
          X2 := MouseX;
        end
        else
        begin
          X1 := MouseX;
          X2 := StartX;
        end;
        if MouseY >= StartY then
        begin
          Y1 := MouseY;
          Y2 := StartY;
        end
        else
        begin
          Y1 := StartY;
          Y2 := MouseY;
        end;
        if MouseY <> StartY then
        begin
          Inc( Y1 );
          Dec( Y2 );
        end;
        X1 := X1 * RatioX - OffsetX;
        X2 := (X2+1) * RatioX - OffsetX;
        Y1 := Y1 * RatioY - OffsetY;
        Y2 := (Y2+1) * RatioY - OffsetY;
        Rectangle( X1, Y1, X2, Y2 );
        with rectSelected do
        begin
          if X1 > X2 then
          begin
            Left := X2;
            Right := X1;
          end
          else
          begin
            Left := X1;
            Right := X2;
          end;
          if Y1 > Y2 then
          begin
            Top := Y2;
            Bottom := Y1;
          end
          else
          begin
            Top := Y1;
            Bottom := Y2;
          end;
          Left := Left + RatioX - 1;
          Top := Top + RatioY - 1;
        end;
      end;
    end;
  end;
end;

(*********************************************
TSquareGridSprite
*********************************************)
procedure TSquareGridSpriteSurface.translateCoords( var X: integer; var Y: integer );
begin
  if RatioX <> 0 then
    X := Trunc( X / RatioX );
  if RatioY <> 0 then
    Y := Trunc( Y / RatioY );
end;

procedure TSquareGridSpriteSurface.drawCursor;
var
  DrawX, DrawY: integer;
begin
  DrawX := (FCursX * RatioX) - OffsetX;
  DrawY := (FCursY * RatioY) - OffsetY;
  DIBCanvas.Rectangle( DrawX, DrawY, DrawX + RatioX, DrawY + RatioY );
  DIBCanvas.Rectangle( DrawX - 1, DrawY - 1, DrawX + RatioX + 1, DrawY + RatioY + 1 );
end;

procedure TSquareGridSpriteSurface.drawMouse;
var
  DrawX, DrawY: integer;
begin
  DrawX := (MouseX * RatioX) - OffsetX;
  DrawY := (MouseY * RatioY) - OffsetY;
  DIBCanvas.Rectangle( DrawX - 3, DrawY - 3, DrawX + RatioX + 3, DrawY + RatioY + 3 );
end;

procedure TSquareGridSpriteSurface.drawGrid;
begin

end;

procedure Register;
begin
  RegisterPropertyEditor( TypeInfo( byte ), TSpriteSurface, 'CursorColorIndex', TColorIndexSelectorSS );
  RegisterPropertyEditor( TypeInfo( byte ), TSpriteSurface, 'MouseColorIndex', TColorIndexSelectorSS );
  RegisterPropertyEditor( TypeInfo( byte ), TSpriteSurface, 'GridColorIndex', TColorIndexSelectorSS );
  RegisterPropertyEditor( TypeInfo( byte ), TSpriteSurface, 'SelectBoxColorIndex', TColorIndexSelectorSS );
  RegisterComponents( 'TurboSprite', [TSquareGridSpriteSurface] );
end;

end.
